from django.shortcuts import render, redirect, get_object_or_404
import logging

# Create your views here.
from django.contrib.auth.decorators import login_required
from django.core.paginator import Paginator
from django.db.models import Q
from django.db import transaction
from django.views.decorators.http import require_POST

from .models import Client
from .forms import (
    ClientForm,
)
from .helpers import (
    extract_people_payload,
    validate_people_payload,
    serialize_people_for_ui,
    save_people_and_legacy_contacts,
)

logger = logging.getLogger(__name__)


@login_required
def client_list(request):
    user_id = request.user.id
    # ---- Pagination size ----
    per_page = request.GET.get("per_page", 10)
    try:
        per_page = int(per_page)
    except ValueError:
        logger.warning("Invalid per_page value=%s from user_id=%s; defaulting to 10", request.GET.get("per_page"), user_id)
        per_page = 10

    if per_page not in [10, 20, 50, 100]:
        logger.warning("Unsupported per_page value=%s from user_id=%s; defaulting to 10", per_page, user_id)
        per_page = 10

    # ---- Filters ----
    query = request.GET.get("q", "").strip()

    clients_qs = Client.objects.filter(is_deleted=False).prefetch_related(
        "names", "emails", "mobiles"
    )

    if query:
        clients_qs = clients_qs.filter(
            Q(names__name__icontains=query) |
            Q(emails__email__icontains=query) |
            Q(mobiles__mobile_number__icontains=query)
        ).distinct()

    # ---- Pagination ----
    paginator = Paginator(clients_qs.order_by("id"), per_page)
    page_number = request.GET.get("page")
    page_obj = paginator.get_page(page_number)
    logger.info(
        "Client list viewed by user_id=%s query=%s per_page=%s page=%s total_results=%s",
        user_id,
        query,
        per_page,
        page_obj.number,
        paginator.count,
    )

    context = {
        "page_obj": page_obj,
        "per_page": per_page,
        "query": query,
        "per_page_options": [10, 20, 50, 100],  
    }

    return render(request, "clients/list.html", context)

@login_required
def client_create(request):
    user_id = request.user.id
    people_initial = [{"name": "", "is_primary": True, "emails": [""], "mobiles": [""]}]
    people_errors = []

    if request.method == "POST":
        logger.info("Client create requested by user_id=%s", user_id)
        client_form = ClientForm(request.POST)
        people_initial = extract_people_payload(request.POST)
        people_errors = validate_people_payload(people_initial)

        if client_form.is_valid() and not people_errors:
            with transaction.atomic():
                client = client_form.save()
                save_people_and_legacy_contacts(client, people_initial)
            logger.info("Client created successfully by user_id=%s client_id=%s", user_id, client.id)

            return redirect("clients:client_index")
        logger.warning(
            "Client create validation failed for user_id=%s client_errors=%s people_errors=%s",
            user_id,
            client_form.errors,
            people_errors,
        )

    else:
        logger.info("Client create form opened by user_id=%s", user_id)
        client_form = ClientForm()

    context = {
        "client_form": client_form,
        "people_initial": people_initial,
        "people_errors": people_errors,
    }
    return render(request, "clients/form.html", context)


@login_required
def client_detail(request, pk):
    client = get_object_or_404(
        Client.objects.filter(is_deleted=False).prefetch_related("persons__emails", "persons__mobiles"),
        pk=pk
    )
    logger.info("Client detail viewed by user_id=%s client_id=%s", request.user.id, client.id)
    return render(request, "clients/detail.html", {"client": client})


@login_required
def client_edit(request, pk):
    client = get_object_or_404(
        Client.objects.prefetch_related("persons__emails", "persons__mobiles"),
        pk=pk,
        is_deleted=False,
    )
    user_id = request.user.id
    people_errors = []

    if request.method == "POST":
        logger.info("Client edit requested by user_id=%s client_id=%s", user_id, client.id)
        client_form = ClientForm(request.POST, instance=client)
        people_initial = extract_people_payload(request.POST)
        people_errors = validate_people_payload(people_initial)

        if client_form.is_valid() and not people_errors:
            with transaction.atomic():
                client_form.save()
                save_people_and_legacy_contacts(client, people_initial)
            logger.info("Client updated successfully by user_id=%s client_id=%s", user_id, client.id)

            return redirect("clients:client_detail", pk=client.pk)
        logger.warning(
            "Client edit validation failed for user_id=%s client_id=%s client_errors=%s people_errors=%s",
            user_id,
            client.id,
            client_form.errors,
            people_errors,
        )

    else:
        logger.info("Client edit form opened by user_id=%s client_id=%s", user_id, client.id)
        client_form = ClientForm(instance=client)
        people_initial = serialize_people_for_ui(client)
        if not people_initial:
            people_initial = [{"name": "", "is_primary": True, "emails": [""], "mobiles": [""]}]

    context = {
        "client_form": client_form,
        "people_initial": people_initial,
        "people_errors": people_errors,
        "client": client,
        "is_edit": True,
    }

    return render(request, "clients/form.html", context)


@login_required
@require_POST
def client_delete(request, pk):
    client = get_object_or_404(Client, pk=pk, is_deleted=False)
    client.is_deleted = True
    client.save(update_fields=["is_deleted"])
    logger.info("Client soft-deleted by user_id=%s client_id=%s", request.user.id, client.id)
    return redirect("clients:client_index")
